<?php

namespace Wb4WpTheme\Managers;

use Wb4WpTheme\Managers\Customize\Customize_Settings;

/**
 * Manages the templates
 * Class Template_Manager
 *
 * @package Wb4WpTheme\Managers
 */
final class Template_Manager {

	const HEADER_TEMPLATE = 'wb4wp_header_section';
	const FOOTER_TEMPLATE = 'wb4wp_footer_section';

	/**
	 * Returns the selected header
	 * the user or a layout provided in $layout
	 *
	 * @param string $setting_name Name of the setting.
	 * @return mixed|null
	 */
	public static function get_header_layout_name( $setting_name ) {
		return self::get_layout_or_get_default(
			$setting_name,
			array(
				'navigation-1',
				'navigation-2',
				'navigation-3',
				'navigation-4',
				'navigation-5',
			)
		);
	}
	/**
	 * Returns the selected header by
	 * the user or a layout provided in $layout
	 *
	 * @return mixed|null
	 */
	public static function get_header() {
		$layout = self::get_header_layout_name( self::HEADER_TEMPLATE );

		$template       = self::get_layout( $layout );
		$meta_data      = get_option( 'wb4wp_metadata', null );
		$global_binding = get_option( 'wb4wp_global_binding', null );

		return self::render(
			$template,
			array(
				'meta_data'      => $meta_data,
				'global_binding' => $global_binding,
				'theme_options'  => array(),
			)
		);
	}

	/**
	 * Get footer layout name
	 *
	 * @param string $setting_name name of the setting.
	 * @return array
	 */
	public static function get_footer_layout_name( $setting_name ) {
		return self::get_layout_or_get_default(
			$setting_name,
			array(
				'footer-1',
				'footer-2',
				'footer-3',
				'footer-4',
				'footer-5',
			)
		);
	}

	/**
	 * Returns the selected footer by
	 * the user or a layout provided in $layout
	 *
	 * @return mixed|null
	 */
	public static function get_footer() {
		$layout = self::get_footer_layout_name( self::FOOTER_TEMPLATE );

		$template       = self::get_layout( $layout );
		$meta_data      = get_option( 'wb4wp_metadata', null );
		$global_binding = get_option( 'wb4wp_global_binding', null );

		return self::render(
			$template,
			array(
				'meta_data'      => $meta_data,
				'global_binding' => $global_binding,
				'theme_options'  => array(),
			)
		);
	}

	/**
	 * Get layout or the default setting.
	 *
	 * @param string $setting_name name of the setting.
	 * @param array  $map .
	 * @return string layout name
	 */
	private static function get_layout_or_get_default( $setting_name, $map ) {
		$layout = Customize_Settings::get_setting( $setting_name . '_layout_setting' );

		if ( ! in_array( $layout, $map, true ) ) {
			$layout = $map[0];
		}

		return $layout;
	}

	/**
	 * Render the template assets.
	 */
	public static function render_template_assets() {

		$header_layout_name = self::get_header_layout_name( self::HEADER_TEMPLATE );
		$footer_layout_name = self::get_footer_layout_name( self::FOOTER_TEMPLATE );

		self::render_asset_by_template( $header_layout_name );
		self::render_asset_by_template( $footer_layout_name );
	}

	/**
	 * Retrieves the version of the theme.
	 */
	public static function get_theme_version() {
		$theme = wp_get_theme();

		return $theme->version;
	}

	/**
	 * Renders the assets by the layout name.
	 *
	 * @param string $layout_name the filename if the asset located in the dist folder.
	 */
	private static function render_asset_by_template( $layout_name ) {
		$style_path = '/dist/' . $layout_name . '/' . $layout_name . '.css';
		$has_style  = file_exists( __dir__ . '/../..' . $style_path );
		$version    = self::get_theme_version();

		if ( $has_style ) {
			wp_enqueue_style( $layout_name, get_template_directory_uri() . $style_path, array(), $version );
		}

		$script_path = '/dist/' . $layout_name . '/' . $layout_name . '.js';
		$has_script  = file_exists( __dir__ . '/../..' . $script_path );

		if ( $has_script ) {
			wp_enqueue_script( $layout_name, get_template_directory_uri() . $script_path, array(), $version, true );
		}
	}

	/**
	 * Generates the layout path
	 * and returns it
	 *
	 * @param string $file_name Name of the file.
	 *
	 * @return string
	 */
	private static function get_layout( $file_name ) {
		$template_dir = get_template_directory();
		return "{$template_dir}/dist/{$file_name}/{$file_name}.php";
	}

	/**
	 * Renders the given template.
	 *
	 * @param string $template path of the template.
	 * @param array  $variables .
	 * @return mixed
	 */
	private static function render( $template, $variables = array() ) {
		$output = '';

		if ( file_exists( $template ) ) {
			// @codingStandardsIgnoreStart
			extract( $variables );
			// @codingStandardsIgnoreEnd

			ob_start();

			include_once $template;

			$output = ob_get_clean();
		}

		return $output;
	}

	/**
	 * Get font name and weight based on setting name
	 *
	 * @param string $full_setting_name
	 * @return void
	 */
	private static function get_font_setting( $full_setting_name ) {
		$setting  = Customize_Settings::get_setting( $full_setting_name );
		$exploded = explode( ':', $setting );
		return array(
			'font'   => $exploded[0],
			'weight' => $exploded[1],
		);
	}

	public static function get_pattern_color( $css_variable_key ) {
		$global_css_variables = self::get_global_css_variables();
		$color                = $global_css_variables[ $css_variable_key ];

		$luminosity = Color_Manager::calculate_luminosity( $color );

		if ( $luminosity > 0.3 ) {
			return 'black';
		} else {
			return 'white';
		}
	}

	public static function get_global_css_variables() {
		$fonts_body_setting    = self::get_font_setting( 'wb4wp_fonts_section_body_setting' );
		$fonts_heading_setting = self::get_font_setting( 'wb4wp_fonts_section_heading_setting' );

		return array(
			'--wb4wp-background'                   => Customize_Settings::get_setting( 'wb4wp_color_section_background_setting' ),
			'--wb4wp-background-stronger'          => Color_Manager::get_background_color_strong(),
			'--wb4wp-background-strongest'         => Color_Manager::get_background_color_stronger(),
			'--wb4wp-background-lighter'           => Color_Manager::get_background_color_lighter(),
			'--wb4wp-background-on-accent2'        => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent2_setting' ),
			'--wb4wp-background-on-accent2-softer' => Color_Manager::get_color_softer( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent2_setting' ) ),
			'--wb4wp-background-on-accent2-10'     => Color_Manager::set_opacity_in_color( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent2_setting' ), 0.1 ),
			'--wb4wp-background-on-accent1'        => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent1_setting' ),
			'--wb4wp-background-on-accent1-softer' => Color_Manager::get_color_softer( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent1_setting' ) ),
			'--wb4wp-background-on-accent1-10'     => Color_Manager::set_opacity_in_color( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_accent2_setting' ), 0.1 ),
			'--wb4wp-background-on-text'           => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_text_setting' ),
			'--wb4wp-background-on-text-10'        => Color_Manager::set_opacity_in_color( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_background_setting', 'wb4wp_color_section_text_setting' ), 0.1 ),

			'--wb4wp-text-color'                   => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_text_setting', null ),
			'--wb4wp-text-color-softer'            => Color_Manager::get_color_softer( null ),
			'--wb4wp-text-color-stronger'          => Color_Manager::get_text_color_stronger(),
			'--wb4wp-text-color-10'                => Color_Manager::get_text_color_with_opacity( 0.1 ),
			'--wb4wp-text-color-20'                => Color_Manager::get_text_color_with_opacity( 0.2 ),
			'--wb4wp-text-color-75'                => Color_Manager::get_text_color_with_opacity( 0.75 ),
			'--wb4wp-text-color-contrast'          => 'rgb(255, 255, 255)',

			'--wb4wp-accent1'                      => Color_Manager::get_background_color_by_name( 'wb4wp_color_section_accent1_setting', null ),
			'--wb4wp-accent1-on-background'        => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent1_setting', 'wb4wp_color_section_background_setting' ),
			'--wb4wp-accent1-on-text'              => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent1_setting', 'wb4wp_color_section_text_setting' ),
			'--wb4wp-text-accent1'                 => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_text_setting', 'wb4wp_color_section_accent1_setting' ),
			'--wb4wp-text-accent1-softer'          => Color_Manager::get_color_softer( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_text_setting', 'wb4wp_color_section_accent1_setting' ) ),
			'--wb4wp-accent1-stronger'             => Color_Manager::get_color_stronger( Color_Manager::get_background_color_by_name( 'wb4wp_color_section_accent1_setting', null ) ),
			'--wb4wp-title-accent1'                => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent2_setting', 'wb4wp_color_section_accent1_setting' ),
			'--wb4wp-title-accent1-text'           => Color_Manager::get_text_color_for_color( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent2_setting', 'wb4wp_color_section_accent1_setting' ) ),

			'--wb4wp-accent2'                      => Color_Manager::get_background_color_by_name( 'wb4wp_color_section_accent2_setting', null ),
			'--wb4wp-text-accent2'                 => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent2_setting', null ),
			'--wb4wp-text-accent2-softer'          => Color_Manager::get_color_softer( Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent2_setting', null ) ),
			'--wb4wp-accent2-stronger'             => Color_Manager::get_color_stronger( Color_Manager::get_background_color_by_name( 'wb4wp_color_section_accent2_setting', null ) ),
			'--wb4wp-title-accent2'                => Color_Manager::get_text_color_on_background( 'wb4wp_color_section_accent2_setting', 'wb4wp_color_section_accent2_setting' ),

			'--wb4wp-primary-color'                => Color_Manager::get_primary_color(),
			'--wb4wp-primary-color-text'           => Color_Manager::get_primary_color_text(),
			'--wb4wp-primary-color-light'          => Color_Manager::get_primary_color_light(),
			'--wb4wp-primary-color-lighter'        => Color_Manager::get_primary_color_lighter(),
			'--wb4wp-primary-color-lightest'       => Color_Manager::get_primary_color_lightest(),
			'--wb4wp-primary-color-border'         => Color_Manager::get_primary_color_border(),
			'--wb4wp-primary-color-stronger'       => Color_Manager::get_primary_color_stronger(),

			'--wb4wp-border-color'                 => Color_Manager::get_border_color(),

			'--wb4wp-font-body'                    => (string) $fonts_body_setting['font'],
			'--wb4wp-font-body-weight'             => $fonts_body_setting['weight'],
			'--wb4wp-font-heading'                 => (string) $fonts_heading_setting['font'],
			'--wb4wp-font-heading-weight'          => $fonts_heading_setting['weight'],
			'--wb4wp-font-size-override'           => ( (float) Customize_Settings::get_setting( 'wb4wp_fonts_section_font_size_setting' ) * 100 ) . '%',
			'--wb4wp-font-size-factor'             => (float) Customize_Settings::get_setting( 'wb4wp_fonts_section_font_size_setting' ),
		);
	}
}
